using System;
using System.Xml;
using System.Net;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// Class represents VistA M server connection information. 
	/// </summary>
	public class ServerConnectionInfo
	{
		/// <summary>
		/// Minimum port number.
		/// </summary>
		public const int MinPort = 1;

		/// <summary>
		/// Maximum port number.
		/// </summary>
		public const int MaxPort = 65534;

		private string _handle = null;
		private string _alias = null;
		private IPAddress _ip = null;
		//private string _domain = null;
		private int _portNumber = IPEndPoint.MinPort;
			
		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_SERVER_CONNECTION_INFO_NODE_NAME = "Server";
		private const string XMLCONSTS_HANDLE_ATTRIBUTE_NAME = "handle";
		private const string XMLCONSTS_ALIAS_ATTRIBUTE_NAME = "alias";
		private const string XMLCONSTS_IP_ATTRIBUTE_NAME = "ip";
		private const string XMLCONSTS_DOMAIN_ATTRIBUTE_NAME = "domain";
		private const string XMLCONSTS_PORT_ATTRIBUTE_NAME = "port";

		private ServerConnectionInfo(){}

		/// <summary>
		/// Primary constructor. Specifies everything. 
		/// </summary>
		/// <param name="handle">Internal handle used to refer to M server in code and config files. </param>
		/// <param name="alias">M server alias</param>
		/// <param name="IP">IP address</param>
		/// <param name="portNumber">M server port number</param>
		public ServerConnectionInfo( string handle, string alias, IPAddress IP, int portNumber )
		{
			if( IP == null ) 
				throw( new ArgumentNullException( "IPAddress" ) ); // CR 2299

			if( portNumber < IPEndPoint.MinPort || portNumber > IPEndPoint.MaxPort )
				throw( new ArgumentOutOfRangeException( "portNumber", SR.Exceptions.ServerConnectionInfoPortNumberIsOutOfRange( IPEndPoint.MinPort, IPEndPoint.MaxPort ) ) );

			_handle = handle;
			_alias = alias;
			_ip = IP;
			_portNumber = portNumber;			
		}

	    /// <summary>
	    /// Primary constructor. Specifies everything. 
	    /// </summary>
	    /// <param name="handle">Internal handle used to refer to M server in code and config files. </param>
	    /// <param name="alias">M server alias.</param>
	    /// <param name="ipString">IP address as string.</param>
	    /// <param name="portNumber">M server port number.</param>
	    public ServerConnectionInfo( string handle, string alias, string ipString, int portNumber ) : 
			this( handle, alias, IPAddress.Parse( ipString ), portNumber )
		{
		}

		/// <summary>
		/// Basic initialization. This constructor only specifies parameters required to connect to M server. 
		/// </summary>
		/// <param name="IP">M server IP address.</param>
		/// <param name="portNumber">M server port number.</param>
		public ServerConnectionInfo( IPAddress IP, int portNumber ) : 
			this( null, null, IP, portNumber ){}

		/// <summary>
		/// Basic initialization. This constructor only specifies parameters required to connect to M server. 
		/// </summary>
		/// <param name="ipString">M server IP address.</param>
		/// <param name="portNumber">M server port number.</param>
		public ServerConnectionInfo( string ipString, int portNumber ) : 
			this( null, null, ipString, portNumber ){}

		/// <summary>
		/// Basic initialization. This constructor only specifies parameters required to connect to M server.
		/// </summary>
		/// <param name="ipString">M server IP address or null.</param>
		/// <param name="domain">M server domain name.</param>
		/// <param name="portNumber">M server port number.</param>
		public ServerConnectionInfo( string ipString, string domain, int portNumber ) :
			this(null, null, ipString, portNumber ){}
		/// <summary>
		/// XML deserialization factory method parsing supplied server connection info
		/// XML node and returning server connection info object created from it.
		/// </summary>
		/// <param name="serverConnectionInfoNode">
		///		Server connection info XML node.
		///	</param>
		/// <returns>Server connection info object deserialized from XML.</returns>
		public static ServerConnectionInfo Parse( XmlNode serverConnectionInfoNode )
		{
			if( serverConnectionInfoNode == null )
				throw( new ArgumentNullException( "serverConnectionInfoNode" ) );

			var sciElement = XmlUtility.ParseValidateConvertNodeToElement( serverConnectionInfoNode, XMLCONSTS_SERVER_CONNECTION_INFO_NODE_NAME );

			try
			{
				return new ServerConnectionInfo( 
					XmlUtility.ParseGetRequiredAttributeValue( sciElement, XMLCONSTS_HANDLE_ATTRIBUTE_NAME ),
					XmlUtility.ParseGetRequiredAttributeValue( sciElement, XMLCONSTS_ALIAS_ATTRIBUTE_NAME ),
					XmlUtility.ParseGetRequiredAttributeValue( sciElement, XMLCONSTS_IP_ATTRIBUTE_NAME ),
					Int32.Parse( XmlUtility.ParseGetRequiredAttributeValue( sciElement, XMLCONSTS_PORT_ATTRIBUTE_NAME ) ) );
			}
			catch( FlatMediumParseException )
			{
				throw;
			}
			catch( Exception xcp ) // have to catch exception here because Int32.Parse and IPAddress.Parse throw various exceptions
			{
				if( !( xcp is ArgumentException || xcp is FormatException || xcp is OverflowException ) )
					throw;

				throw( new XmlParseException( SR.Exceptions.XmlDeserializationFailedDueToInnerException( 
					typeof(ServerConnectionInfo).Name, sciElement.Name ), xcp ) );				
			}
		}

		/// <summary>
		/// Serializes server connection info to a first child XML element of a given node. 
		/// Will update existing element if found with the same handle or create a new one. 
		/// </summary>
		/// <param name="parentNode">Parent XML node.</param>
		/// <returns>Resulting <see cref="XmlElement"/> data was serialized to.</returns>
		public XmlElement ToChildXmlElement( XmlNode parentNode )
		{
			if( parentNode == null )
				throw( new ArgumentNullException( "parentNode" ) );

			var sourceNode = parentNode.SelectSingleNode( 
				String.Format( "descendant::{0}[@{1}='{2}']", XMLCONSTS_SERVER_CONNECTION_INFO_NODE_NAME, XMLCONSTS_HANDLE_ATTRIBUTE_NAME, _handle  ) );

			if( sourceNode == null )
				sourceNode = parentNode.InsertBefore( parentNode.OwnerDocument.CreateElement( XMLCONSTS_SERVER_CONNECTION_INFO_NODE_NAME ), parentNode.FirstChild );

			var sourceElement = (XmlElement)sourceNode;

			sourceElement.SetAttribute( XMLCONSTS_HANDLE_ATTRIBUTE_NAME, _handle );
			sourceElement.SetAttribute( XMLCONSTS_ALIAS_ATTRIBUTE_NAME, _alias );
			sourceElement.SetAttribute( XMLCONSTS_IP_ATTRIBUTE_NAME, _ip.ToString() );
			sourceElement.SetAttribute( XMLCONSTS_PORT_ATTRIBUTE_NAME, _portNumber.ToString() );

			return sourceElement;
		}

		/// <summary>
		/// Override of standard method checking for equality of two objects.  This is called
		/// when a user changes divisions.  If everything is equal, the user isn't prompted 
		/// to re-enter their Access and Verify codes.
		/// </summary>
		/// <param name="obj">Objects to check equality for.</param>
		/// <returns>True if supplied object is equivalent to this object.</returns>
		public override bool Equals( object obj )
		{
			if( obj == null || GetType() != obj.GetType() ) 
				return false;

			var sci = (ServerConnectionInfo)obj;

			return 
				CheckEquals( this.Alias, sci.Alias ) && 
				CheckEquals( this.PortNumber, sci.PortNumber ) && 
				CheckEquals( this.Handle, sci.Handle ) &&
				CheckDomainAndIPEquality(this.IPAddress, sci.IPAddress); //CR 2299
		}

		private bool CheckEquals( object obj1, object obj2 )
		{
			return obj1 == null ? false : obj1.Equals( obj2 );			
		}

	    /// <summary>
	    /// Checks to see if either the domain name or IPs are equal.
	    /// </summary>
	    /// <param name="IP"></param>
	    /// <param name="IPTwo"></param>
	    /// <returns>bool indicating equality of either domain or IP fields</returns>
	    /// CR 2299
	    private bool CheckDomainAndIPEquality(object IP, object IPTwo)
		{
			if (IP != null && IPTwo != null)
			{
				if (IP.Equals(IPTwo)) 
				{
					return true;
				}
				else
				{
					return false;
				}
			}
			else
			{
				return false;
			}
		}


		/// <summary>
		/// Override of standard method getting hash code (required because override to <see cref="Equals"/> is added).
		/// </summary>
		/// <returns>Object hash code.</returns>
		public override int GetHashCode()
		{
			return base.GetHashCode ();
		}

		/// <summary>
		/// Internal handle used to refer to M server in code and config files. 
		/// This handle is to be hardcoded and put in config file. 
		/// It is to be used to identify M server. 
		/// Optional value. 
		/// </summary>
		public string Handle
		{
			get
			{
				return _handle;
			}
		}

		/// <summary>
		/// M server alias. Optional value. 
		/// </summary>
		public string Alias
		{
			get
			{
				return _alias;
			}
		}

		/// <summary>
		/// IP address of the M server.
		/// </summary>
		public IPAddress IPAddress
		{
			get
			{
				return _ip;
			}
		}

		/// <summary>
		/// M server port number. 
		/// </summary>
		public int PortNumber
		{
			get
			{
				return _portNumber;
			}
		}
	}
}
